"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteFeaturedItem = exports.updateFeaturedItem = exports.createFeaturedItem = exports.getFeaturedCollections = void 0;
const client_1 = require("@prisma/client");
const prisma = new client_1.PrismaClient();
// GET /api/storefront/collections/featured
const getFeaturedCollections = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const collections = yield prisma.featuredCollection.findMany({
            where: { active: true },
            orderBy: { order: 'asc' },
            include: {
                product: {
                    select: {
                        name: true,
                        slug: true,
                        price: true,
                        images: true, // Assuming this is string[]
                        category: { select: { slug: true } }
                    }
                }
            }
        });
        // Transform to match the UI structure
        const formattedData = collections.map(item => {
            // Safety check in case product was deleted but relation persists (though Cascade handles this)
            if (!item.product)
                return null;
            return {
                id: item.id,
                title: item.title, // The custom label from FeaturedCollection
                productName: item.product.name, // The actual product name
                // Format price (assuming INR based on previous context)
                subtitle: `$ ${Number(item.product.price).toLocaleString('en-US')}`,
                image: item.product.images[0] || '/placeholder.jpg',
                // Construct link: /category-slug/product-slug
                link: `/${item.product.category.slug}/${item.product.slug}`
            };
        }).filter(Boolean); // Remove nulls
        res.json(formattedData);
    }
    catch (error) {
        console.error("Featured Collections Error:", error);
        res.status(500).json({ error: "Failed to fetch featured collections" });
    }
});
exports.getFeaturedCollections = getFeaturedCollections;
// POST /api/admin/featured
const createFeaturedItem = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { title, productId, order, active } = req.body;
        if (!title || !productId) {
            return res.status(400).json({ error: "Title and Product are required" });
        }
        const newItem = yield prisma.featuredCollection.create({
            data: {
                title,
                productId,
                order: parseInt(order) || 0,
                active: active !== undefined ? active : true
            }
        });
        res.status(201).json(newItem);
    }
    catch (error) {
        console.error("Create Featured Error:", error);
        res.status(500).json({ error: "Failed to create item" });
    }
});
exports.createFeaturedItem = createFeaturedItem;
// PUT /api/admin/featured/:id
const updateFeaturedItem = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { id } = req.params;
        const { title, productId, order, active } = req.body;
        const updated = yield prisma.featuredCollection.update({
            where: { id },
            data: {
                title,
                productId,
                order: parseInt(order),
                active
            }
        });
        res.json(updated);
    }
    catch (error) {
        res.status(500).json({ error: "Failed to update item" });
    }
});
exports.updateFeaturedItem = updateFeaturedItem;
// DELETE /api/admin/featured/:id
const deleteFeaturedItem = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { id } = req.params;
        yield prisma.featuredCollection.delete({ where: { id } });
        res.json({ message: "Item deleted" });
    }
    catch (error) {
        res.status(500).json({ error: "Failed to delete item" });
    }
});
exports.deleteFeaturedItem = deleteFeaturedItem;
